import {Request, Response} from 'express';
import {responseJSON} from '../../helpers/response-handler';
import {loginSvc, registerSvc} from './authPanel.service';
import UserPanel from '../../db/models/UserPanel';
import RolePanel from '../../db/models/RolePanel';
import RoleAppPanel from '../../db/models/RoleAppPanel';

// interface AuthRequest extends Request {
//   user?: {id: string}; // Sesuaikan dengan struktur user
// }

interface AuthRequest extends Request {
  user?: UserSubPayload;
}
// Tipe User dari token
export interface UserSubPayload {
  sub: {
    id: number;
    name: string;
    username: string;
    role_id: number;
    createdAt: string;
    updatedAt: string;
  };
  iss: string;
  iat: number;
  exp: number;
}

export const register = async (req: Request, res: Response) => {
  try {
    const user = await registerSvc(req.body);
    return responseJSON(req, res, {data: user});
  } catch (error) {
    console.error(error);
    return responseJSON(req, res, error, true);
  }
};

export const login = async (req: Request, res: Response) => {
  try {
    const result = await loginSvc(req.body);
    return res.json({
      status: 'SUCCESS',
      message: 'Success',
      ...result,
    });
  } catch (error) {
    return responseJSON(req, res, error, true);
  }
};

export const profile = async (req: AuthRequest, res: Response) => {
  try {
    console.log('Masuk ke endpoint profile');
    console.log('Decoded JWT user: ', req?.user);

    const userPayload = req?.user as UserSubPayload;
    const userId = userPayload?.sub?.id;

    if (!userId) {
      return res.status(400).json({message: 'User ID not found in token'});
    }

    const userInstance = await UserPanel.findOne({
      where: { id: userId },
      include: [
        {
          model: RolePanel,
          as: 'role',
          include: [
            {
              model: RoleAppPanel,
              as: 'apps',
              attributes: ['id','app_id'], 
            },
          ],
        },
      ],
    });

    if (!userInstance) {
      return res.status(404).json({message: 'User not found'});
    }

    const user = userInstance.get({plain: true});

    if (!user) {
      return res.status(404).json({message: 'User not found'});
    }

    return res.json({
      message: 'Success',
      user: {
        id: user.id,
        name: user.name,
        username: user.username,
        role: {
          id: user.role?.id || null,
          name: user.role?.name || null,
          slug: user.role?.slug || null,
          createdAt: user.role?.createdAt,
          updatedAt: user.role?.updatedAt,
          apps: user.role?.apps || [],
        },
      },
    });
  } catch (error) {
    console.error('Error fetching profile:', error);
    return res.status(500).json({message: 'Internal server error'});
  }
};

// export const refreshToken = async (req: Request, res: Response) => {
//   try {
//     const refreshToken = req.body.refreshToken;
//     const decoded = jwt.verify(refreshToken, REFRESH_JWT_SECRET);
//     const token = jwt.sign(
//       {
//         userId: decoded.userId,
//         email: decoded.email,
//       },
//       JWT_SECRET,
//       {expiresIn: JWT_EXPIRES_IN},
//     );
//     return res.json({message: 'Token refreshed', token});
//   } catch (error) {
//     return responseJSON(req, res, error, true);
//   }
// };
