import {
  getAllUsersRpo,
  getAllAdminsRpo,
  getAdminByUsername,
  getClientById,
  createAdmin,
} from './auth.repository';

import {responseJSON} from '../../helpers/response-handler';

const jwt = require('jsonwebtoken');
const bcrypt = require('bcrypt');

export const getAllUsersSvc = async () => {
  return await getAllUsersRpo();
};

export const getAllAdminsSvc = async () => {
  return await getAllAdminsRpo();
};

//V1

export const getTokenSvc = async (username: String, key: String) => {
  let clientSecret = '';
  const adminData = await getAdminByUsername(username);
  if (!adminData) {
    return {error: {message: 'Invalid Admin username', code: 10012}};
  }

  const isSame = await bcrypt.compare(key, adminData.password);
  if (!isSame) {
    return {error: {message: 'Signature verification failed', code: 10012}};
  }

  if (adminData.client === 'ADMIN') {
    clientSecret = process.env.ADMIN_CLIENT_SECRET;
  } else {
    const clientCheck = await getClientById(adminData.client);
    if (!clientCheck) {
      return {error: {message: 'Invalid app_id', code: 10012}};
    }
    clientSecret = clientCheck.client_secret;
  }
  console.log('Token befiore make it', adminData);
  const tokenData = jwt.sign(
    {
      appid: adminData.client,
      username: adminData.username,
      email: adminData.email,
    },
    clientSecret,
    {
      expiresIn: '3h',
    },
  );

  return {tokenData};
};

export const operatorSignUpSvc = async (
  appid: string,
  username: string,
  email: string,
  key: string,
) => {
  const hashedPassword = await bcrypt.hash(key, 11);

  const data = {
    username,
    email,
    password: hashedPassword,
    client: appid,
  };

  const userData = await createAdmin(data);

  if (!userData) {
    return {error: {message: 'Failed to create admin', code: 10012}};
  }

  const token = jwt.sign(
    {
      id: data.username,
      client_id: userData.client,
      name: username,
    },
    process.env.JWT_SECRET as string,
    {expiresIn: '3h'},
  );

  return {user: userData, tokenData: token};
};

//V2

export const profileSvc = async (username: string) => {
  
  const adminData = await getAdminByUsername(username);
  if (!adminData) {
    return {error: {message: 'Username not found', code: 10012}};
  }

  const {password, ...safeData} = adminData.get({plain: true}); // Convert Sequelize instance to plain object

  return {data: safeData};
};
