import {build} from 'joi';
import {
  getAllClientsRpo,
  findClientById,
  createClient,
  findPartnerByName,
  createAllBuckets,
  findAdminByUsername,
  updateAdminPassword,
} from './client.repository';
import crypto from 'crypto';
import sequelize from '../../db/config';
import bcrypt from 'bcrypt';

let currenciesData = [
  {
    code: 'usd',
    name: 'United State Dollar',
    symbol: '$',
    min_bet: 1,
    max_bet: 1000,
    bet_category_1: 2,
    bet_category_2: 10,
    bet_category_3: 50,
    bet_category_4: 200,
    decimal_digits: 2,
    idr_rates: 16573.2203,
    usd_rates: 1,
  },
  {
    code: 'idr',
    name: 'Indonesian Rupiah',
    symbol: 'Rp',
    min_bet: 200,
    max_bet: 1500000,
    bet_category_1: 100,
    bet_category_2: 1000,
    bet_category_3: 10000,
    bet_category_4: 100000,
    decimal_digits: 2,
    idr_rates: 1,
    usd_rates: 0.000061,
  },
];

export const getAllClientsSvc = async () => {
  return await getAllClientsRpo();
};

export const clientSignupSvc = async (data: any) => {
  const {
    appid,
    client_name,
    partner_name,
    client_secret,
    bucket_currency,
    bucket_max,
  } = data;

  const t = await sequelize.transaction(); // 👈 start transaction
  try {
    const existingClient = await findClientById(appid);
    if (existingClient) {
      await t.rollback();
      return {error: {message: 'Duplicate Account', code: 10017}};
    }

    const partner = await findPartnerByName(partner_name);
    if (!partner || !partner.id) {
      await t.rollback();
      return {error: {message: 'Partner does not exist', code: 10013}};
    }

    const secret = client_secret || crypto.randomBytes(16).toString('hex');

    const clientData = await createClient({
      id: appid,
      name: client_name,
      total_user: 0,
      partner_id: partner.id,
      join_date: new Date(),
      client_username: '-',
      client_password: '-',
      client_link_auth: '-',
      client_link_verify: '-',
      client_link_get_balance: '-',
      excluded_game: '-',
      client_secret: secret,
      shared: 0,
    });

    const max = Number(bucket_max) || 10000000;

    let currencyChoice = 'idr';
    if (bucket_currency) {
      const currencyCheck = currenciesData.find(
        (c) => c.code === bucket_currency,
      );
      currencyChoice = currencyCheck ? bucket_currency : 'idr';
    }

    await createAllBuckets(appid, currencyChoice, max); // 👈 modify this to accept transaction

    await t.commit(); // ✅ all good
    return {client_data: clientData};
  } catch (error) {
    await t.rollback(); // ❌ undo everything
    return {error: {message: error.message}};
  }
};

export const clientPasswordChangeSvc = async (data:any) => {
  const {username, newPassword} = data;
  const daAdmin = await findAdminByUsername(username);

  if (!daAdmin) {
    return { error: { message: 'Invalid username', code: 10009 } };
  }

  const hashedPassword = await bcrypt.hash(newPassword, 11);
  await updateAdminPassword(daAdmin, hashedPassword);

  return { code: 1, msg: 'success' };
};
