import {
  getDropdownsRpo,
  getYearReportsRpo,
  getMonthReportRpo,
  getDailyReportRpo,
  getMonthlyGameReportRpo,
} from './home.repository';

import dayjs from 'dayjs'; // kalau kamu pakai dayjs

export interface PlatformSummary {
  year: string; // atau `number`, tergantung hasil dari SQL
  total: number;
}

const generateLast7Days = (): string[] => {
  const today = new Date();
  const dates: string[] = [];

  for (let i = 0; i < 7; i++) {
    const date = new Date(today);
    date.setDate(today.getDate() - i);

    const yyyy = date.getFullYear();
    const mm = String(date.getMonth() + 1).padStart(2, '0');
    const dd = String(date.getDate()).padStart(2, '0');

    dates.push(`${yyyy}-${mm}-${dd}`);
  }

  return dates;
};

function generateMonthList(start: string, end: string): string[] {
  const [startYear, startMonth] = start.split('-').map(Number);
  const [endYear, endMonth] = end.split('-').map(Number);

  const months: string[] = [];
  let year = startYear;
  let month = startMonth;

  while (year < endYear || (year === endYear && month <= endMonth)) {
    months.push(`${year}-${String(month).padStart(2, '0')}`);
    month++;
    if (month > 12) {
      month = 1;
      year++;
    }
  }

  return months;
}

// Langkah 1: generate array tanggal (format 'YYYY-MM-DD') antara startDate dan endDate
const generateDateList = (start: string, end: string): string[] => {
  const result = [];
  let current = new Date(start);
  const last = new Date(end);

  while (current <= last) {
    result.push(current.toISOString().slice(0, 10)); // 'YYYY-MM-DD'
    current.setDate(current.getDate() + 1);
  }

  return result;
};

export const getWinAndLoseReportsSvc = async (id, currency) => {
  const now = new Date();

  // 📅 Daily Summary
  // const todayStr = now.toISOString().split('T')[0];
  // const yesterdayStr = new Date(now.setDate(now.getDate() - 1))
  //   .toISOString()
  //   .split('T')[0];

  // const yesterdayStr="2025-06-12"
  // const todayStr="2025-06-13"

  const todayStr = dayjs().format('YYYY-MM-DD');
  const yesterdayStr = dayjs().subtract(1, 'day').format('YYYY-MM-DD');

  const dateList = [todayStr, yesterdayStr];

  const todayData = await getDailyReportRpo({
    currency,
    id,
    dateList,
  });

  const todayTotal = todayData.find((d) => d.trandate === todayStr)?.total || 0;
  const yesterdayTotal =
    todayData.find((d) => d.trandate === yesterdayStr)?.total || 0;

  let dayChange = 0;
  if (yesterdayTotal === 0) {
    dayChange = todayTotal === 0 ? 0 : 100;
  } else {
    dayChange =
      ((todayTotal - yesterdayTotal) / Math.abs(yesterdayTotal)) * 100;
  }

  const dayChangeText =
    dayChange === Infinity
      ? '+∞%'
      : dayChange === -Infinity
      ? '-∞%'
      : `${dayChange >= 0 ? '+' : ''}${dayChange.toFixed(2)}%`;

  const wl_day = {
    // todayTotal,
    // yesterdayTotal:yesterdayTotal,
    change: isFinite(dayChange)
      ? parseFloat(dayChange.toFixed(2))
      : dayChangeText,
    total: todayTotal,
  };

  // 📆 Monthly Summary
  const endMonth = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(
    2,
    '0',
  )}`;
  const startMonthDate = new Date(now.getFullYear(), now.getMonth() - 1, 1);
  const startMonth = `${startMonthDate.getFullYear()}-${String(
    startMonthDate.getMonth() + 1,
  ).padStart(2, '0')}`;

  const monthList = [startMonth, endMonth]; // bisa juga pakai fungsi generate range jika lebih dari 2 bulan

  const monthlyData = await getMonthReportRpo({
    currency,
    id,
    monthList,
  });

  const endTotal = monthlyData.find((d) => d.trandate === endMonth)?.total || 0;

  const startMonthTotal =
    monthlyData.find((d) => d.trandate === startMonth)?.total || 0;

  let monthChange = 0;
  if (startMonthTotal === 0) {
    monthChange = endTotal > 0 ? Infinity : endTotal < 0 ? -Infinity : 0;
  } else {
    monthChange =
      ((endTotal - startMonthTotal) / Math.abs(startMonthTotal)) * 100;
  }

  const monthChangeText =
    monthChange === Infinity
      ? '+∞%'
      : monthChange === -Infinity
      ? '-∞%'
      : `${monthChange >= 0 ? '+' : ''}${monthChange.toFixed(2)}%`;

  const wl_month = {
    // month:endTotal,
    // prevmont:previousMonthTotal,
    change: isFinite(monthChange)
      ? parseFloat(monthChange.toFixed(2))
      : monthChangeText,
    total: endTotal,
  };

  const endYear = now.getFullYear().toString();
  const startYear = (now.getFullYear() - 1).toString();

  const yearList = [startYear, endYear]; // array sesuai format RPO yang baru
  const yearlyData = await getYearReportsRpo({
    currency,
    id,
    yearList,
  });

  const endYearTotal =
    yearlyData.find((d) => d.trandate === endYear)?.total || 0;

  const startYearTotal =
    yearlyData.find((d) => d.trandate === startYear)?.total || 0;

  let yearChange = 0;

  if (startYearTotal === 0) {
    yearChange = endYearTotal > 0 ? Infinity : endYearTotal < 0 ? -Infinity : 0;
  } else {
    yearChange =
      ((endYearTotal - startYearTotal) / Math.abs(startYearTotal)) * 100;
  }

  const yearChangeText =
    yearChange === Infinity
      ? '+∞%'
      : yearChange === -Infinity
      ? '-∞%'
      : `${yearChange >= 0 ? '+' : ''}${yearChange.toFixed(2)}%`;

  const wl_year = {
    // previousYearTotal: previousYearTotal,
    // endYearTotal: endYearTotal,
    change: isFinite(yearChange)
      ? parseFloat(yearChange.toFixed(2))
      : yearChangeText,
    total: endYearTotal,
  };

  return {wl_day, wl_month, wl_year};
};

export const getDropdownsSvc = async ({roleId}) => {
  const roleAppData = await getDropdownsRpo({roleId});

  const clients = roleAppData
    .map((item) => item.client)
    .filter((client) => !!client);

  return clients;
};

export const getWinAndLoseByFiltersSvc = async (
  id,
  currency = 'idr',
  filter_by,
) => {
  const now = new Date();
  const endMonth = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(
    2,
    '0',
  )}`;
  const startDate = new Date(now);
  startDate.setMonth(startDate.getMonth() - 7);
  const startMonth = `${startDate.getFullYear()}-${String(
    startDate.getMonth() + 1,
  ).padStart(2, '0')}`;
  const monthList = generateMonthList(startMonth, endMonth);

  let rawData;

  switch (filter_by) {
    case 'daily':
      const dateList = generateLast7Days();
      rawData = await getDailyReportRpo({currency, id, dateList});
      break;

    case 'monthly':
      rawData = await getMonthReportRpo({currency, id, monthList});
      break;

    case 'yearly':
      const startYear = 2000;
      const endYear = now.getFullYear();
      const yearList = Array.from({length: endYear - startYear + 1}, (_, i) =>
        (startYear + i).toString(),
      );
      rawData = await getYearReportsRpo({currency, id, yearList});
      break;

    default:
      throw new Error(
        'Invalid filter_by value. Expected "daily", "monthly", or "yearly".',
      );
  }

  // === Transform hasil ===
  const sorted = [...rawData].sort(
    (a, b) => new Date(a.trandate).getTime() - new Date(b.trandate).getTime(),
  );
  const labels = sorted.map((item) => item.trandate);

  const nameMap = {
    GGSOFT01: 'GG SOFT 1',
    // tambahkan mapping lainnya jika perlu
  };

  const grouped = {};
  for (const item of sorted) {
    const displayName = nameMap[item.name] || item.name;
    if (!grouped[displayName]) grouped[displayName] = [];
    grouped[displayName].push(item.total);
  }

  const client_master = Object.entries(grouped).map(([name, data]) => ({
    name,
    data,
  }));

  return {
    data: {
      labels,
      client_master,
    },
  };
};

export const getWinAndLoseGamesSvc = async (
  id,
  currency = 'idr',
) => {

  const now = new Date();
  const endMonth = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(
    2,
    '0',
  )}`;
  const startMonthDate = new Date(now.getFullYear(), now.getMonth() - 1, 1);
  const startMonth = `${startMonthDate.getFullYear()}-${String(
    startMonthDate.getMonth() + 1,
  ).padStart(2, '0')}`;

  const monthList = [startMonth, endMonth]; // bisa juga pakai fungsi generate range jika lebih dari 2 bulan

  const monthlyData = await getMonthlyGameReportRpo({
    currency,
    id,
    monthList,
  });
  console.log('monthlyData', monthlyData)
  return monthlyData;
};
