import Joi from 'joi';
import {Request, Response} from 'express';
import {responseJSON} from '../../helpers/response-handler';
import {
  createMenuService,
  destroyMenuService,
  getAllMenuService,
  getMenuByIdService,
  updateMenuByIdService,
  getMenusWithSubmenusService,
  getMenuByRoleIdService,
  updateMenuRolesService,
} from './menuPanel.service';
import {formatMenuByRole} from './menuPanel.dto';

export interface AuthenticatedRequest extends Request {
  user?: any;
  id?: string;
  roleId?: string;
}

export const index = async (req: Request, res: Response) => {
  try {
    const page = parseInt(req.query.p as string, 10) || 1;
    const limit = parseInt(req.query.l as string, 10) || 10;

    const pagination =
      req.query.pg === 'true' || req.query.pg === '1'
        ? true
        : req.query.pg === 'false' || req.query.pg === '0'
        ? false
        : true; // default to true if undefined or invalid

    const result = await getAllMenuService(page, limit, pagination);

    return responseJSON(req, res, result);
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

export const tree = async (req: Request, res: Response) => {
  try {
    const result = await getMenusWithSubmenusService();
    return responseJSON(req, res, {data: result});
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

export const getMenuByRoleIdController = async (
  req: Request,
  res: Response,
) => {
  try {
    const result = await getMenuByRoleIdService(Number(req.params.id));
    const formatted = formatMenuByRole(result);
    return responseJSON(req, res, {data: formatted});
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

export const store = async (req: Request, res: Response) => {
  try {
    const schema = Joi.object({
      title: Joi.string().required(),
      icon: Joi.string().allow(null, ''),
      page_name: Joi.string().allow(null, ''),
      parent_id: Joi.number().integer().allow(null),
    });

    const {error} = schema.validate(req.body);

    if (error) {
      return responseJSON(req, res, {message: error.message}, true);
    }

    const result = await createMenuService(req.body);

    return responseJSON(req, res, {data: result});
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

export const show = async (req: Request, res: Response) => {
  try {
    const {id} = req.params;
    const result = await getMenuByIdService(Number(id));

    return responseJSON(req, res, {data: result});
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

export const update = async (req: Request, res: Response) => {
  try {
    const schema = Joi.object({
      title: Joi.string().required(),
      icon: Joi.string().allow(null, ''),
      page_name: Joi.string().allow(null, ''),
      parent_id: Joi.number().integer().allow(null),
    });

    const {error} = schema.validate(req.body);

    if (error) {
      return responseJSON(req, res, {message: error.message}, true);
    }

    const {id} = req.params;
    const result = await updateMenuByIdService(Number(id), req.body);

    return responseJSON(req, res, {data: result});
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

export const destroy = async (req: Request, res: Response) => {
  try {
    const {id} = req.params;

    const result = await destroyMenuService(Number(id));

    return responseJSON(req, res, {data: result});
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

export const getMenuByAuth = async (
  req: AuthenticatedRequest,
  res: Response,
) => {
  try {
    const roleId = req.roleId; // Ambil dari middleware jwt
    const result = await getMenuByRoleIdService(Number(roleId));
    const formatted = formatMenuByRole(result);
    return responseJSON(req, res, {data: formatted});
  } catch (error: any) {
    return responseJSON(req, res, {message: error.message}, true);
  }
};

const updateMenuRolesSchema = Joi.object({
  role_id: Joi.number().required(),
  menu_id: Joi.array().items(Joi.number().required()).min(1).required(),
}).required().strict();


export const updateMenuRoles = async (req: Request, res: Response) => {
  try {
    const { error, value } = updateMenuRolesSchema.validate(req.body, {
      abortEarly: false,
    });

    if (error) {
      const messages = error.details.map((d: any) => d.message).join(', ');

      return res.status(400).json({
        status: 'FAILED',
        message: messages,
        code: 400,
        data: {
          validationErrors: error.details,
        },
      });
    }

    const { role_id, menu_id } = value;
    const result = await updateMenuRolesService(role_id, menu_id);

    return res.status(200).json({
      status: 'SUCCESS',
      message: 'Success',
      data: result,
    });
  } catch (error: any) {
    return res.status(500).json({
      status: 'FAILED',
      message: error.message || 'Internal server error',
      code: 500,
    });
  }
};
