import MenuPanel, {MenuPanelAttributes} from '../../db/models/MenuPanel';
import MenuRolePanel, {
  MenuRolePanelAttributes,
} from '../../db/models/MenuRolePanel';
import {Op} from 'sequelize';

export const getAllMenuRepository = async (
  page: number,
  limit: number,
  pagination: boolean,
) => {
  try {
    if (pagination) {
      const offset = (page - 1) * limit;

      const {count, rows} = await MenuPanel.findAndCountAll({
        offset,
        limit,
        // ✅ only add subQuery if you’re including associations
        // remove it if you're not using `include`
      });
      console.log('Returned rows:', rows.length);

      const totalPages = Math.ceil(count / limit);

      return {
        data: rows.map((item) => item.toJSON()),
        pagination: {
          totalRecords: count,
          totalPages,
          currentPage: page,
        },
      };
    } else {
      const {count, rows} = await MenuPanel.findAndCountAll();

      return {
        data: rows.map((item) => item.toJSON()),
        totalRecords: count,
      };
    }
  } catch (error) {
    throw error;
  }
};

export const getMenuByRoleIdRepository = async (roleId: number) => {
  try {
    const result = await MenuRolePanel.findAll({
      where: {role_id: roleId},

      include: [
        {
          model: MenuPanel,
          as: 'menu',
          attributes: [
            'id',
            'title',
            'icon',
            'page_name',
            'parent_id',
            'is_active',
          ],
          where: {
            is_active: true, // Tambahkan kondisi ini
          },
        },
      ],
      order: [
        ['menu', 'id', 'ASC'], // Urutkan berdasarkan id menu
        ['menu', 'parent_id', 'ASC'], // Urutkan berdasarkan parent_id
      ],
    });

    return result.map((row) => row.toJSON());
  } catch (error) {
    throw error;
  }
};

export const getMenuByIdRepository = async (id: number) => {
  try {
    const menu = await MenuPanel.findByPk(id);
    return menu?.toJSON();
  } catch (error) {
    throw error;
  }
};

export const createMenuRepository = async (
  data: Partial<MenuPanelAttributes>,
) => {
  try {
    const menu = await MenuPanel.create(data);
    return menu.toJSON();
  } catch (error) {
    throw error;
  }
};

export const updateMenuByIdRepository = async (
  id: number,
  updateData: Partial<MenuPanelAttributes>,
) => {
  try {
    const menu = await MenuPanel.findByPk(id);

    if (!menu) {
      throw new Error(`Menu not found`);
    }

    await menu.update(updateData);
    return menu.toJSON();
  } catch (error) {
    throw error;
  }
};

export const destroyMenuByIdRepository = async (id: number) => {
  try {
    const menu = await MenuPanel.findByPk(id);

    if (!menu) {
      throw new Error(`Menu with id ${id} not found`);
    }

    await menu.destroy();
    return menu.toJSON();
  } catch (error) {
    throw error;
  }
};

export const getAllMenusWithSubmenus = async () => {
  try {
    const allMenus = await MenuPanel.findAll();
    const menus = allMenus.map((m) => m.toJSON());

    const mainMenus = menus.filter((m) => !m.parent_id);
    const subMenus = menus.filter((m) => m.parent_id);

    const structured = mainMenus.map((menu) => {
      const subMenu = subMenus.filter((sm) => sm.parent_id === menu.id);
      return {
        ...menu,
        subMenu: subMenu.length ? subMenu : undefined,
      };
    });

    return structured;
  } catch (error) {
    throw error;
  }
};

export const updateMenusRoleRelationship = async (roleId: number, menuIds: number[]) => {
  const transaction = await MenuRolePanel.sequelize!.transaction();

  try {
    // Step 1: Delete existing relations for the given roleId
    await MenuRolePanel.destroy({
      where: { role_id: roleId },
      transaction,
    });

    // Step 2: Create new relations
    const newRelations = menuIds.map((menuId) => ({
      role_id: roleId,
      menu_id: menuId,
    }));

    await MenuRolePanel.bulkCreate(newRelations, { transaction });

    await transaction.commit();
    return { success: true };
  } catch (error) {
    await transaction.rollback();
    throw error;
  }
};
