import { Request, Response } from 'express';
import * as Joi from 'joi';
import { responseJSON } from '../../helpers/response-handler';
import {
  getRolePanelService,
  getRolePanelByIdService,
  createRolePanelService,
  updateRolePanelService,
  deleteRolePanelService,
} from './rolepanel.service';
import logger from '../../helpers/logger';

export const getRolePanelController = async (req: Request, res: Response) => {
  try {
    logger.info('getRolePanelController - Request received', {
      page: req.query.page,
      pageSize: req.query.pageSize,
      search: req.query.search,
    });

    const page = parseInt(req.query.page as string, 10) || 1;
    const pageSize = parseInt(req.query.pageSize as string, 10) || 10;
    const search = req.query.search as string || '';

    const roles = await getRolePanelService(page, pageSize, search);

    logger.info('getRolePanelController - Successfully fetched roles', {
      count: roles?.length || 0,
    });

    return responseJSON(req, res, { data: roles });
  } catch (error) {
    logger.error('getRolePanelController - Error', {
      error: error.message,
      stack: error.stack,
    });

    return responseJSON(req, res, error, true);
  }
};

export const getRolePanelByIdController = async (req: Request, res: Response) => {
    try {
      const roleId = req.params.id;
      logger.info('getRolePanelByIdController - Request received', { roleId });
  
      const role = await getRolePanelByIdService(roleId);
  
      if (!role) {
        logger.warn('getRolePanelByIdController - Role not found', { roleId });
        return res.status(404).json({
          status: 'FAILED',
          message: 'Not Found',
          data: {},
        });
      }
      
      return responseJSON(req, res, { data: role });
    } catch (error) {
      logger.error('getRolePanelByIdController - Error', {
        error: error.message,
        stack: error.stack,
      });
  
      return responseJSON(req, res, error, true);
    }
  };

export const createRolePanelController = async (req: Request, res: Response) => {
  try {
    const schema = Joi.object({
      name: Joi.string().required(),
      slug: Joi.string().required(),
    });

    const { error, value } = schema.validate(req.body);
    if (error) {
      return responseJSON(req, res, { message: error.message }, true);
    }

    const role = await createRolePanelService(value);
    return responseJSON(req, res, { data: role });
  } catch (error) {
    logger.error('createRolePanelController - Error', {
      error: error.message,
      stack: error.stack,
    });

    return responseJSON(req, res, error, true);
  }
};

export const updateRolePanelController = async (req: Request, res: Response) => {
  try {
    const schema = Joi.object({
      name: Joi.string().optional(),
      slug: Joi.string().optional(),
    });

    const { error, value } = schema.validate(req.body);
    if (error) {
      return responseJSON(req, res, { message: error.message }, true);
    }

    const roleId = req.params.id;
    const updated = await updateRolePanelService(roleId, value);
    return responseJSON(req, res, { data: updated });
  } catch (error) {
    logger.error('updateRolePanelController - Error', {
      error: error.message,
      stack: error.stack,
    });

    return responseJSON(req, res, error, true);
  }
};

export const deleteRolePanelController = async (req: Request, res: Response) => {
  try {
    const roleId = req.params.id;
    const deleted = await deleteRolePanelService(roleId);
    return responseJSON(req, res, { data: deleted });
  } catch (error) {
    logger.error('deleteRolePanelController - Error', {
      error: error.message,
      stack: error.stack,
    });

    return responseJSON(req, res, error, true);
  }
};
