import { Op } from 'sequelize';
import RolePanel, { RolePanelAttributes } from '../../db/models/RolePanel';

interface PaginationParams {
  page: number;
  pageSize: number;
  search?: string;
}

// Get list of roles with pagination and optional search
export const getRolePanelRepository = async (
  where: RolePanelAttributes,
  { page, pageSize, search }: PaginationParams
) => {
  const offset = (page - 1) * pageSize;

  const searchCondition = search ? {
    [Op.or]: [
      { name: { [Op.iLike]: `%${search}%` } },
      { slug: { [Op.iLike]: `%${search}%` } },
    ]
  } : {};

  const finalWhere = { ...where, ...searchCondition };

  const roles = await RolePanel.findAll({
    where: finalWhere,
    limit: pageSize,
    offset,
    order: [['id', 'ASC']],
  });

  return roles.map((role) => role.toJSON());
};

// Create a new role
export const createRolePanelRepository = async (data: RolePanelAttributes) => {
  const newRole = await RolePanel.create(data);
  return newRole.toJSON();
};

// Update role by ID
export const updateRolePanelRepository = async (
  roleId: string,
  data: Partial<RolePanelAttributes>
) => {
  const role = await RolePanel.findByPk(roleId);

  if (!role) return null;

  await role.update(data);
  return role.toJSON();
};

// Get role by ID
export const getRolePanelByIdRepository = async (roleId: string) => {
  const role = await RolePanel.findByPk(roleId);
  return role ? role.toJSON() : null;
};

// Delete role by ID
export const deleteRolePanelRepository = async (roleId: string) => {
  const role = await RolePanel.findByPk(roleId);
  if (!role) return null;

  await role.destroy();
  return roleId;
};
